const SoundCloud = {
  CLIENT_ID: 'iuGz93EpTX3C4QRivhkmkNNvEVru9xoY',
  CLIENT_SECRET: '0uwANvNM3Pq78OmWQ4qOEqHEMkqQnTwZ',
  API_BASE: 'https://api.soundcloud.com',
  AUTH_URL: 'https://secure.soundcloud.com/authorize',
  TOKEN_URL: 'https://secure.soundcloud.com/oauth/token',

  accessToken: null,
  user: null,

  getRedirectURL() {
    return chrome.identity.getRedirectURL('callback');
  },

  async authorize() {
    const redirectUri = this.getRedirectURL();
    console.log('[SoundCloud] Redirect URI:', redirectUri);

    const authUrl = `${this.AUTH_URL}?` + new URLSearchParams({
      client_id: this.CLIENT_ID,
      redirect_uri: redirectUri,
      response_type: 'code',
      scope: 'non-expiring'
    }).toString();

    console.log('[SoundCloud] Auth URL:', authUrl);

    return new Promise((resolve, reject) => {
      console.log('[SoundCloud] Launching web auth flow...');
      chrome.identity.launchWebAuthFlow(
        { url: authUrl, interactive: true },
        async (responseUrl) => {
          console.log('[SoundCloud] Callback received');
          console.log('[SoundCloud] Response URL:', responseUrl);
          console.log('[SoundCloud] Last error:', chrome.runtime.lastError);

          if (chrome.runtime.lastError) {
            console.error('[SoundCloud] Auth error:', chrome.runtime.lastError.message);
            reject(new Error(chrome.runtime.lastError.message));
            return;
          }

          if (!responseUrl) {
            console.error('[SoundCloud] No response URL received');
            reject(new Error('No response URL'));
            return;
          }

          const url = new URL(responseUrl);
          const code = url.searchParams.get('code');
          console.log('[SoundCloud] Authorization code:', code ? 'received' : 'missing');

          if (!code) {
            const error = url.searchParams.get('error_description') || 'Authorization failed';
            console.error('[SoundCloud] Auth failed:', error);
            reject(new Error(error));
            return;
          }

          try {
            console.log('[SoundCloud] Exchanging code for token...');
            const tokenData = await this.exchangeCodeForToken(code, redirectUri);
            console.log('[SoundCloud] Token received successfully');
            this.accessToken = tokenData.access_token;
            await this.saveToken(tokenData);
            resolve(tokenData);
          } catch (err) {
            console.error('[SoundCloud] Token exchange failed:', err);
            reject(err);
          }
        }
      );
    });
  },

  async exchangeCodeForToken(code, redirectUri) {
    console.log('[SoundCloud] Exchanging code for token...');
    console.log('[SoundCloud] Using redirect URI:', redirectUri);

    const response = await fetch(this.TOKEN_URL, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
      },
      body: new URLSearchParams({
        client_id: this.CLIENT_ID,
        client_secret: this.CLIENT_SECRET,
        grant_type: 'authorization_code',
        code: code,
        redirect_uri: redirectUri
      }).toString()
    });

    console.log('[SoundCloud] Token response status:', response.status);

    if (!response.ok) {
      const error = await response.text();
      console.error('[SoundCloud] Token exchange error:', error);
      throw new Error(`Token exchange failed: ${error}`);
    }

    const tokenData = await response.json();
    console.log('[SoundCloud] Token data received:', JSON.stringify(tokenData, null, 2));
    return tokenData;
  },

  async saveToken(tokenData) {
    await chrome.storage.local.set({
      soundcloud_token: tokenData.access_token,
      soundcloud_refresh: tokenData.refresh_token
    });
  },

  async loadToken() {
    const data = await chrome.storage.local.get(['soundcloud_token']);
    if (data.soundcloud_token) {
      this.accessToken = data.soundcloud_token;
      return true;
    }
    return false;
  },

  async logout() {
    this.accessToken = null;
    this.user = null;
    await chrome.storage.local.remove(['soundcloud_token', 'soundcloud_refresh']);
  },

  isAuthenticated() {
    return !!this.accessToken;
  },

  async apiRequest(endpoint, options = {}) {
    if (!this.accessToken) {
      throw new Error('Not authenticated');
    }

    const url = endpoint.startsWith('http') ? endpoint : `${this.API_BASE}${endpoint}`;

    console.log('[SoundCloud] API Request:', url);
    console.log('[SoundCloud] Token:', this.accessToken ? this.accessToken.substring(0, 20) + '...' : 'none');

    const response = await fetch(url, {
      ...options,
      headers: {
        'Accept': 'application/json',
        'Authorization': `Bearer ${this.accessToken}`,
        ...options.headers
      }
    });

    console.log('[SoundCloud] API Response status:', response.status);

    if (!response.ok) {
      const errorText = await response.text();
      console.error('[SoundCloud] API Error:', errorText);
      if (response.status === 401) {
        await this.logout();
        throw new Error('Session expired. Please login again.');
      }
      throw new Error(`API request failed: ${response.status}`);
    }

    return response.json();
  },

  async getMe() {
    const user = await this.apiRequest('/me');
    this.user = user;
    return user;
  },

  async getLikes(limit = 50, offset = 0) {
    return this.apiRequest(`/me/likes?limit=${limit}&offset=${offset}`);
  },

  async getPlaylists() {
    return this.apiRequest('/me/playlists');
  },

  async getPlaylistTracks(playlistId) {
    return this.apiRequest(`/playlists/${playlistId}`);
  },

  async getMyTracks() {
    return this.apiRequest('/me/tracks');
  },

  async search(query, limit = 20) {
    return this.apiRequest(`/tracks?q=${encodeURIComponent(query)}&limit=${limit}`);
  },

  async getStreamUrl(trackId) {
    console.log('[SoundCloud] Getting stream URL for track:', trackId);

    // Use v2 API for track details - should work with new secure OAuth tokens
    const v2Url = `https://api-v2.soundcloud.com/tracks/${trackId}?client_id=${this.CLIENT_ID}`;
    console.log('[SoundCloud] Fetching from v2 API:', v2Url);

    const response = await fetch(v2Url, {
      headers: {
        'Accept': 'application/json',
        'Authorization': `OAuth ${this.accessToken}`
      }
    });

    console.log('[SoundCloud] v2 API response status:', response.status);

    if (!response.ok) {
      console.error('[SoundCloud] v2 API failed:', response.status);
      throw new Error('Failed to get track details');
    }

    const track = await response.json();
    console.log('[SoundCloud] Track data from v2:', track);

    // Use media transcodings from v2 API
    if (track.media && track.media.transcodings && track.media.transcodings.length > 0) {
      console.log('[SoundCloud] Found transcodings:', track.media.transcodings.length);

      // Find progressive stream (direct MP3, works with audio element)
      const progressive = track.media.transcodings.find(t =>
        t.format && t.format.protocol === 'progressive'
      );
      // Fallback to HLS if no progressive
      const transcoding = progressive || track.media.transcodings[0];
      console.log('[SoundCloud] Using transcoding:', transcoding.format);

      if (transcoding && transcoding.url) {
        // Fetch the actual stream URL
        const streamResponse = await fetch(`${transcoding.url}?client_id=${this.CLIENT_ID}`, {
          headers: {
            'Authorization': `OAuth ${this.accessToken}`
          }
        });

        if (streamResponse.ok) {
          const streamData = await streamResponse.json();
          console.log('[SoundCloud] Stream data:', streamData);
          if (streamData && streamData.url) {
            console.log('[SoundCloud] Got stream URL!');
            return streamData.url;
          }
        } else {
          console.error('[SoundCloud] Stream fetch failed:', streamResponse.status);
        }
      }
    }

    console.error('[SoundCloud] No stream available for track');
    throw new Error('Track is not streamable');
  },

  formatDuration(ms) {
    const seconds = Math.floor(ms / 1000);
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  },

  formatArtwork(url, size = 't300x300') {
    if (!url) return null;
    return url.replace('-large', `-${size}`);
  }
};

if (typeof module !== 'undefined' && module.exports) {
  module.exports = SoundCloud;
}
