class MusicPlayer {
  constructor() {
    this.widget = null;
    this.widgetIframe = document.getElementById('scWidget');
    this.playlist = [];
    this.currentIndex = -1;
    this.isPlaying = false;
    this.isShuffle = false;
    this.repeatMode = 0;
    this.currentTab = 'likes';
    this.playlists = [];
    this.widgetDuration = 0;

    this.initElements();
    this.initWidget();
    this.initEventListeners();
    this.checkAuth();
  }

  initWidget() {
    // Initialize SoundCloud Widget API
    if (typeof SC !== 'undefined' && SC.Widget) {
      this.widget = SC.Widget(this.widgetIframe);

      this.widget.bind(SC.Widget.Events.READY, () => {
        console.log('[Widget] Ready');
      });

      this.widget.bind(SC.Widget.Events.PLAY, () => {
        this.isPlaying = true;
        this.updatePlayButton();
      });

      this.widget.bind(SC.Widget.Events.PAUSE, () => {
        this.isPlaying = false;
        this.updatePlayButton();
      });

      this.widget.bind(SC.Widget.Events.FINISH, () => {
        this.handleTrackEnd();
      });

      this.widget.bind(SC.Widget.Events.PLAY_PROGRESS, (data) => {
        if (this.widgetDuration > 0) {
          const percent = (data.currentPosition / this.widgetDuration) * 100;
          this.progressBar.value = percent;
          this.updateProgressFill(percent);
          this.currentTimeEl.textContent = this.formatTime(data.currentPosition / 1000);
        }
      });
    } else {
      console.error('[Widget] SoundCloud Widget API not loaded');
    }
  }

  initElements() {
    // Screens
    this.loginScreen = document.getElementById('loginScreen');
    this.appScreen = document.getElementById('appScreen');
    this.loginBtn = document.getElementById('loginBtn');
    this.logoutBtn = document.getElementById('logoutBtn');

    // Search
    this.searchBtn = document.getElementById('searchBtn');
    this.searchPanel = document.getElementById('searchPanel');
    this.searchInput = document.getElementById('searchInput');
    this.searchClose = document.getElementById('searchClose');
    this.searchResults = document.getElementById('searchResults');



    // Player controls
    this.playBtn = document.getElementById('playBtn');
    this.prevBtn = document.getElementById('prevBtn');
    this.nextBtn = document.getElementById('nextBtn');
    this.shuffleBtn = document.getElementById('shuffleBtn');
    this.repeatBtn = document.getElementById('repeatBtn');
    this.progressBar = document.getElementById('progressBar');
    this.progressFill = document.getElementById('progressFill');
    this.volumeSlider = document.getElementById('volumeSlider');
    this.currentTimeEl = document.getElementById('currentTime');
    this.durationEl = document.getElementById('duration');
    this.trackNameEl = document.getElementById('trackName');
    this.artistNameEl = document.getElementById('artistName');
    this.albumArtEl = document.getElementById('albumArt');
    this.playIcon = this.playBtn.querySelector('.play-icon');
    this.pauseIcon = this.playBtn.querySelector('.pause-icon');
  }

  initEventListeners() {
    // Auth
    this.loginBtn.addEventListener('click', () => this.login());
    this.logoutBtn.addEventListener('click', () => this.logout());

    // Search
    this.searchBtn.addEventListener('click', () => this.toggleSearch());
    this.searchClose.addEventListener('click', () => this.toggleSearch());
    this.searchInput.addEventListener('input', this.debounce(() => this.performSearch(), 300));



    // Player controls
    this.playBtn.addEventListener('click', () => this.togglePlay());
    this.prevBtn.addEventListener('click', () => this.playPrevious());
    this.nextBtn.addEventListener('click', () => this.playNext());
    this.shuffleBtn.addEventListener('click', () => this.toggleShuffle());
    this.repeatBtn.addEventListener('click', () => this.toggleRepeat());

    // Progress bar - seek in widget
    this.progressBar.addEventListener('input', (e) => {
      if (this.widget && this.widgetDuration > 0) {
        const time = (e.target.value / 100) * this.widgetDuration;
        this.widget.seekTo(time);
      }
      this.updateProgressFill(e.target.value);
    });

    // Volume - control widget volume
    this.volumeSlider.addEventListener('input', (e) => {
      if (this.widget) {
        this.widget.setVolume(e.target.value);
      }
    });
  }

  async checkAuth() {
    const hasToken = await SoundCloud.loadToken();
    if (hasToken) {
      try {
        await SoundCloud.getMe();
        this.showApp();
      } catch (err) {
        this.showLogin();
      }
    } else {
      this.showLogin();
    }
  }

  async login() {
    this.loginBtn.disabled = true;
    this.loginBtn.innerHTML = '<span class="material-symbols-outlined spinning">progress_activity</span> Connecting...';

    try {
      await SoundCloud.authorize();
      await SoundCloud.getMe();
      this.showApp();
    } catch (err) {
      console.error('Login error:', err);
      this.showError('Login failed: ' + err.message);
    } finally {
      this.loginBtn.disabled = false;
      this.loginBtn.innerHTML = '<span class="material-symbols-outlined">login</span> Sign in with SoundCloud';
    }
  }

  async logout() {
    await SoundCloud.logout();
    this.playlist = [];
    this.currentIndex = -1;
    if (this.widget) {
      this.widget.pause();
    }
    this.showLogin();
  }

  showLogin() {
    this.loginScreen.classList.remove('hidden');
    this.appScreen.classList.add('hidden');
  }

  showApp() {
    this.loginScreen.classList.add('hidden');
    this.appScreen.classList.remove('hidden');
  }

  toggleSearch() {
    this.searchPanel.classList.toggle('hidden');
    if (!this.searchPanel.classList.contains('hidden')) {
      this.searchInput.focus();
    } else {
      this.searchInput.value = '';
      this.searchResults.innerHTML = '';
    }
  }

  async performSearch() {
    const query = this.searchInput.value.trim();
    if (!query) {
      this.searchResults.innerHTML = '';
      return;
    }

    this.searchResults.innerHTML = '<div class="loading"><span class="material-symbols-outlined spinning">progress_activity</span></div>';

    try {
      const tracks = await SoundCloud.search(query);
      this.renderSearchResults(tracks);
    } catch (err) {
      this.searchResults.innerHTML = '<p class="error">Search failed</p>';
    }
  }

  renderSearchResults(tracks) {
    if (!tracks || tracks.length === 0) {
      this.searchResults.innerHTML = '<p class="no-results">No tracks found</p>';
      return;
    }

    this.searchResults.innerHTML = tracks.map((track, index) => this.renderTrackItem(track, index, 'search')).join('');
    this.addTrackListeners(this.searchResults, tracks, 'search');
  }





  renderTrackItem(track, index, source) {
    const artwork = track.artwork_url ? SoundCloud.formatArtwork(track.artwork_url, 't67x67') : null;
    const duration = SoundCloud.formatDuration(track.duration);
    const artist = track.user?.username || 'Unknown Artist';

    return `
      <div class="track-item" data-index="${index}" data-source="${source}">
        <div class="track-item-art">
          ${artwork ? `<img src="${artwork}" alt="">` : '<span class="material-symbols-outlined">music_note</span>'}
        </div>
        <div class="track-item-info">
          <div class="track-item-title">${track.title}</div>
          <div class="track-item-artist">${artist}</div>
        </div>
        <div class="track-item-duration">${duration}</div>
      </div>
    `;
  }

  addTrackListeners(container, tracks, source) {
    container.querySelectorAll('.track-item').forEach(item => {
      item.addEventListener('click', () => {
        const index = parseInt(item.dataset.index);
        this.playlist = tracks;
        this.loadTrack(index);
        // Close search panel if open
        if (source === 'search' && !this.searchPanel.classList.contains('hidden')) {
          this.toggleSearch();
        }
      });
    });
  }

  async loadTrack(index) {
    if (index < 0 || index >= this.playlist.length) return;

    this.currentIndex = index;
    const track = this.playlist[index];

    this.trackNameEl.textContent = track.title;
    this.artistNameEl.textContent = track.user?.username || 'Unknown Artist';

    // Update album art
    if (track.artwork_url) {
      const artworkUrl = SoundCloud.formatArtwork(track.artwork_url, 't300x300');
      this.albumArtEl.innerHTML = `<img src="${artworkUrl}" alt="">`;
    } else {
      this.albumArtEl.innerHTML = '<span class="material-symbols-outlined album-placeholder">album</span>';
    }

    // Load track into widget using permalink_url or construct URL from ID
    const trackUrl = track.permalink_url || `https://api.soundcloud.com/tracks/${track.id}`;
    console.log('[Player] Loading track into widget:', trackUrl);

    if (this.widget) {
      this.widget.load(trackUrl, {
        auto_play: true,
        show_artwork: false,
        callback: () => {
          console.log('[Player] Track loaded, getting duration');
          this.widget.getDuration((duration) => {
            this.widgetDuration = duration;
            this.durationEl.textContent = this.formatTime(duration / 1000);
          });
        }
      });
    } else {
      console.error('[Player] Widget not initialized');
      this.showError('Player not ready');
    }
  }

  togglePlay() {
    if (this.playlist.length === 0) return;

    if (this.currentIndex === -1) {
      this.loadTrack(0);
      return;
    }

    if (this.widget) {
      if (this.isPlaying) {
        this.widget.pause();
      } else {
        this.widget.play();
      }
    }
  }

  updatePlayButton() {
    if (this.isPlaying) {
      this.playIcon.style.display = 'none';
      this.pauseIcon.style.display = 'block';
    } else {
      this.playIcon.style.display = 'block';
      this.pauseIcon.style.display = 'none';
    }
  }

  playPrevious() {
    if (this.playlist.length === 0) return;

    let newIndex;
    if (this.isShuffle) {
      newIndex = Math.floor(Math.random() * this.playlist.length);
    } else {
      newIndex = this.currentIndex - 1;
      if (newIndex < 0) newIndex = this.playlist.length - 1;
    }

    this.loadTrack(newIndex);
  }

  playNext() {
    if (this.playlist.length === 0) return;

    let newIndex;
    if (this.isShuffle) {
      newIndex = Math.floor(Math.random() * this.playlist.length);
    } else {
      newIndex = this.currentIndex + 1;
      if (newIndex >= this.playlist.length) newIndex = 0;
    }

    this.loadTrack(newIndex);
  }

  handleTrackEnd() {
    if (this.repeatMode === 2) {
      // Repeat one - replay same track
      this.loadTrack(this.currentIndex);
    } else if (this.repeatMode === 1 || this.currentIndex < this.playlist.length - 1) {
      this.playNext();
    } else {
      this.isPlaying = false;
      this.updatePlayButton();
    }
  }

  toggleShuffle() {
    this.isShuffle = !this.isShuffle;
    this.shuffleBtn.classList.toggle('active', this.isShuffle);
  }

  toggleRepeat() {
    this.repeatMode = (this.repeatMode + 1) % 3;
    this.repeatBtn.classList.toggle('active', this.repeatMode > 0);

    const icon = this.repeatBtn.querySelector('.material-symbols-outlined');
    if (this.repeatMode === 2) {
      icon.textContent = 'repeat_one';
    } else {
      icon.textContent = 'repeat';
    }
  }

  updateProgress() {
    // Widget handles progress via PLAY_PROGRESS event
  }

  updateProgressFill(percent) {
    this.progressFill.style.width = `${percent}%`;
  }

  updateDuration() {
    // Widget handles duration via getDuration callback
  }

  formatTime(seconds) {
    if (isNaN(seconds)) return '0:00';
    const mins = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  }

  showError(message) {
    console.error(message);
  }

  debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
      const later = () => {
        clearTimeout(timeout);
        func(...args);
      };
      clearTimeout(timeout);
      timeout = setTimeout(later, wait);
    };
  }
}

document.addEventListener('DOMContentLoaded', () => {
  new MusicPlayer();
});
