import base64
import zlib
import struct

def create_png(width, height):
    def make_chunk(chunk_type, data):
        chunk = chunk_type + data
        crc = zlib.crc32(chunk) & 0xffffffff
        return struct.pack('>I', len(data)) + chunk + struct.pack('>I', crc)
    
    # PNG signature
    signature = b'\x89PNG\r\n\x1a\n'
    
    # IHDR chunk
    ihdr_data = struct.pack('>IIBBBBB', width, height, 8, 6, 0, 0, 0)
    ihdr = make_chunk(b'IHDR', ihdr_data)
    
    # Create pixel data with gradient and music note
    raw_data = b''
    cx, cy = width // 2, height // 2
    r = min(width, height) // 2 - 2
    
    for y in range(height):
        raw_data += b'\x00'  # Filter byte
        for x in range(width):
            dx, dy = x - cx, y - cy
            dist = (dx*dx + dy*dy) ** 0.5
            
            if dist <= r:
                # Gradient from purple to pink
                t = (x + y) / (width + height)
                red = int(139 + (236 - 139) * t)
                green = int(92 + (72 - 92) * t)
                blue = int(246 + (153 - 246) * t)
                
                # Simple music note shape
                note_x, note_y = x - width * 0.35, y - height * 0.25
                in_stem = (0 <= note_x <= width * 0.08 and 0 <= note_y <= height * 0.5)
                in_head = ((note_x + width * 0.04) ** 2 + (note_y - height * 0.45) ** 2) <= (height * 0.12) ** 2
                in_flag = (0 <= note_x <= width * 0.25 and 0 <= note_y <= height * 0.15)
                
                if in_stem or in_head or in_flag:
                    raw_data += bytes([255, 255, 255, 255])
                else:
                    raw_data += bytes([red, green, blue, 255])
            else:
                raw_data += bytes([0, 0, 0, 0])
    
    # Compress the data
    compressed = zlib.compress(raw_data, 9)
    idat = make_chunk(b'IDAT', compressed)
    
    # IEND chunk
    iend = make_chunk(b'IEND', b'')
    
    return signature + ihdr + idat + iend

# Create icons
for size in [16, 48, 128]:
    png_data = create_png(size, size)
    with open(f'icon{size}.png', 'wb') as f:
        f.write(png_data)
    print(f'Created icon{size}.png')

